/***************************************************************
 *  806x disassembler - disassembly routines
 *
 *  source file -->  "dis_806x.c"
 *
 *  Copyright   1997,1998   William Lawrance
 ***************************************************************/
#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    "dis_806x.h"

/***************************************************************
 *  the opcode table
 ***************************************************************/
OPCTBLE     opctbl[]    =   {
  1,0,  DI                  ,"skip  ",
  2,1,  DI      +WR         ,"clrw  ",
  2,1,  DI      +WR         ,"cplw  ",
  2,1,  DI      +WR         ,"negw  ",/* 003 */
  1,0,UK                    ,"      ",
  2,1,  DI      +WR         ,"decw  ",
  2,1,  DI      +WR         ,"sexw  ",
  2,1,  DI      +WR         ,"incw  ",/* 007 */
  3,2,  DI               +SH,"shrw  ",
  3,2,  DI               +SH,"shlw  ",
  3,2,  DI               +SH,"asrw  ",
  1,0,UK                    ,"      ",/* 00B */
  3,2,  DI               +SH,"shrdw ",
  3,2,  DI               +SH,"shldw ",
  3,2,  DI               +SH,"asrdw ",
  3,2,  DI                  ,"norml ",/* 00F */
  1,0,UK                    ,"      ",
  2,1,  DI      +WR         ,"clrb  ",
  2,1,  DI      +WR         ,"cplb  ",
  2,1,  DI      +WR         ,"negb  ",/* 013 */
  1,0,UK                    ,"      ",
  2,1,  DI      +WR         ,"decb  ",
  2,1,  DI      +WR         ,"sexb  ",
  2,1,  DI      +WR         ,"incb  ",/* 017 */
  3,2,  DI               +SH,"shrb  ",
  3,2,  DI               +SH,"shlb  ",
  3,2,  DI               +SH,"asrb  ",
  1,0,UK                    ,"      ",/* 01B */
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",/* 01F */
  2,1,  DI                  ,"sjmp  ",
  2,1,  DI                  ,"sjmp  ",
  2,1,  DI                  ,"sjmp  ",
  2,1,  DI                  ,"sjmp  ",/* 023 */
  2,1,  DI                  ,"sjmp  ",
  2,1,  DI                  ,"sjmp  ",
  2,1,  DI                  ,"sjmp  ",
  2,1,  DI                  ,"sjmp  ",/* 027 */
  2,1,  DI                  ,"scall ",
  2,1,  DI                  ,"scall ",
  2,1,  DI                  ,"scall ",
  2,1,  DI                  ,"scall ",/* 02B */
  2,1,  DI                  ,"scall ",
  2,1,  DI                  ,"scall ",
  2,1,  DI                  ,"scall ",
  2,1,  DI                  ,"scall ",/* 02F */
  3,1,  DI                  ,"jbc   ",
  3,1,  DI                  ,"jbc   ",
  3,1,  DI                  ,"jbc   ",
  3,1,  DI                  ,"jbc   ",/* 033 */
  3,1,  DI                  ,"jbc   ",
  3,1,  DI                  ,"jbc   ",
  3,1,  DI                  ,"jbc   ",
  3,1,  DI                  ,"jbc   ",/* 037 */
  3,1,  DI                  ,"jbs   ",
  3,1,  DI                  ,"jbs   ",
  3,1,  DI                  ,"jbs   ",
  3,1,  DI                  ,"jbs   ",/* 03B */
  3,1,  DI                  ,"jbs   ",
  3,1,  DI                  ,"jbs   ",
  3,1,  DI                  ,"jbs   ",
  3,1,  DI                  ,"jbs   ",/* 03F */
  4,3,  DI                  ,"an3w  ",
  5,3,    IM       +IW      ,"an3w  ",
  4,3,      ID              ,"an3w  ",
  5,3,        IX            ,"an3w  ",/* 043 */
  4,3,  DI                  ,"ad3w  ",
  5,3,    IM       +IW      ,"ad3w  ",
  4,3,      ID              ,"ad3w  ",
  5,3,        IX            ,"ad3w  ",/* 047 */
  4,3,  DI                  ,"sb3w  ",
  5,3,    IM       +IW      ,"sb3w  ",
  4,3,      ID              ,"sb3w  ",
  5,3,        IX            ,"sb3w  ",/* 04B */
  4,3,  DI            +PX   ,"ml3w  ",
  5,3,    IM       +IW+PX   ,"ml3w  ",
  4,3,      ID        +PX   ,"ml3w  ",
  5,3,        IX      +PX   ,"ml3w  ",/* 04F */
  4,3,  DI                  ,"an3b  ",
  4,3,    IM                ,"an3b  ",
  4,3,      ID              ,"an3b  ",
  5,3,        IX            ,"an3b  ",/* 053 */
  4,3,  DI                  ,"ad3b  ",
  4,3,    IM                ,"ad3b  ",
  4,3,      ID              ,"ad3b  ",
  5,3,        IX            ,"ad3b  ",/* 057 */
  4,3,  DI                  ,"sb3b  ",
  4,3,    IM                ,"sb3b  ",
  4,3,      ID              ,"sb3b  ",
  5,3,        IX            ,"sb3b  ",/* 05B */
  4,3,  DI            +PX   ,"ml3b  ",
  4,3,    IM          +PX   ,"ml3b  ",
  4,3,      ID        +PX   ,"ml3b  ",
  5,3,        IX      +PX   ,"ml3b  ",/* 05F */
  3,2,  DI                  ,"an2w  ",
  4,2,    IM       +IW      ,"an2w  ",
  3,2,      ID              ,"an2w  ",
  4,2,        IX            ,"an2w  ",/* 063 */
  3,2,  DI                  ,"ad2w  ",
  4,2,    IM       +IW      ,"ad2w  ",
  3,2,      ID              ,"ad2w  ",
  4,2,        IX            ,"ad2w  ",/* 067 */
  3,2,  DI                  ,"sb2w  ",
  4,2,    IM       +IW      ,"sb2w  ",
  3,2,      ID              ,"sb2w  ",
  4,2,        IX            ,"sb2w  ",/* 06B */
  3,2,  DI            +PX   ,"ml2w  ",
  4,2,    IM       +IW+PX   ,"ml2w  ",
  3,2,      ID        +PX   ,"ml2w  ",
  4,2,        IX      +PX   ,"ml2w  ",/* 06F */
  3,2,  DI                  ,"an2b  ",
  3,2,    IM                ,"an2b  ",
  3,2,      ID              ,"an2b  ",
  4,2,        IX            ,"an2b  ",/* 073 */
  3,2,  DI                  ,"ad2b  ",
  3,2,    IM                ,"ad2b  ",
  3,2,      ID              ,"ad2b  ",
  4,2,        IX            ,"ad2b  ",/* 077 */
  3,2,  DI                  ,"sb2b  ",
  3,2,    IM                ,"sb2b  ",
  3,2,      ID              ,"sb2b  ",
  4,2,        IX            ,"sb2b  ",/* 07B */
  3,2,  DI            +PX   ,"ml2b  ",
  3,2,    IM          +PX   ,"ml2b  ",
  3,2,      ID        +PX   ,"ml2b  ",
  4,2,        IX      +PX   ,"ml2b  ",/* 07F */
  3,2,  DI                  ,"orw   ",
  4,2,    IM       +IW      ,"orw   ",
  3,2,      ID              ,"orw   ",
  4,2,        IX            ,"orw   ",/* 083 */
  3,2,  DI                  ,"xrw   ",
  4,2,    IM       +IW      ,"xrw   ",
  3,2,      ID              ,"xrw   ",
  4,2,        IX            ,"xrw   ",/* 087 */
  3,2,  DI                  ,"cmpw  ",
  4,2,    IM       +IW      ,"cmpw  ",
  3,2,      ID              ,"cmpw  ",
  4,2,        IX            ,"cmpw  ",/* 08B */
  3,2,  DI            +PX   ,"divw  ",
  4,2,    IM       +IW+PX   ,"divw  ",
  3,2,      ID        +PX   ,"divw  ",
  4,2,        IX      +PX   ,"divw  ",/* 08F */
  3,2,  DI                  ,"orb   ",
  3,2,    IM                ,"orb   ",
  3,2,      ID              ,"orb   ",
  4,2,        IX            ,"orb   ",/* 093 */
  3,2,  DI                  ,"xorb  ",
  3,2,    IM                ,"xorb  ",
  3,2,      ID              ,"xorb  ",
  4,2,        IX            ,"xorb  ",/* 097 */
  3,2,  DI                  ,"cmpb  ",
  3,2,    IM                ,"cmpb  ",
  3,2,      ID              ,"cmpb  ",
  4,2,        IX            ,"cmpb  ",/* 09B */
  3,2,  DI            +PX   ,"divb  ",
  3,2,    IM          +PX   ,"divb  ",
  3,2,      ID        +PX   ,"divb  ",
  4,2,        IX      +PX   ,"divb  ",/* 09F */
  3,2,  DI                  ,"ldw   ",
  4,2,    IM       +IW      ,"ldw   ",
  3,2,      ID              ,"ldw   ",
  4,2,        IX            ,"ldw   ",/* 0A3 */
  3,2,  DI                  ,"adcw  ",
  4,2,    IM       +IW      ,"adcw  ",
  3,2,      ID              ,"adcw  ",
  4,2,        IX            ,"adcw  ",/* 0A7 */
  3,2,  DI                  ,"sbbw  ",
  4,2,    IM       +IW      ,"sbbw  ",
  3,2,      ID              ,"sbbw  ",
  4,2,        IX            ,"sbbw  ",/* 0AB */
  3,2,  DI                  ,"ldzbw ",
  3,2,    IM                ,"ldzbw ",
  3,2,      ID              ,"ldzbw ",
  4,2,        IX            ,"ldzbw ",/* 0AF */
  3,2,  DI                  ,"ldb   ",
  3,2,    IM                ,"ldb   ",
  3,2,      ID              ,"ldb   ",
  4,2,        IX            ,"ldb   ",/* 0B3 */
  3,2,  DI                  ,"adcb  ",
  3,2,    IM                ,"adcb  ",
  3,2,      ID              ,"adcb  ",
  4,2,        IX            ,"adcb  ",/* 0B7 */
  3,2,  DI                  ,"sbbb  ",
  3,2,    IM                ,"sbbb  ",
  3,2,      ID              ,"sbbb  ",
  4,2,        IX            ,"sbbb  ",/* 0BB */
  3,2,  DI                  ,"ldsbw ",
  3,2,    IM                ,"ldsbw ",
  3,2,      ID              ,"ldsbw ",
  4,2,        IX            ,"ldsbw ",/* 0BF */
  3,2,  DI                  ,"stw   ",
  1,0,UK                    ,"      ",
  3,2,      ID              ,"stw   ",
  4,2,        IX            ,"stw   ",/* 0C3 */
  3,2,  DI                  ,"stb   ",
  1,0,UK                    ,"      ",
  3,2,      ID              ,"stb   ",
  4,2,        IX            ,"stb   ",/* 0C7 */
  2,1,  DI                  ,"pushw ",
  3,1,    IM       +IW      ,"pushw ",
  2,1,      ID              ,"pushw ",
  3,1,        IX            ,"pushw ",/* 0CB */
  2,1,  DI                  ,"popww ",
  1,0,UK                    ,"      ",
  2,1,      ID              ,"popw  ",
  3,1,        IX            ,"popw  ",/* 0CF */
  2,1,  DI                  ,"jnst  ",
  2,1,  DI                  ,"jleu  ",
  2,1,  DI                  ,"jgt   ",
  2,1,  DI                  ,"jnc   ",/* 0D3 */
  2,1,  DI                  ,"jnvt  ",
  2,1,  DI                  ,"jnv   ",
  2,1,  DI                  ,"jge   ",
  2,1,  DI                  ,"jne   ",/* 0D7 */
  2,1,  DI                  ,"jst   ",
  2,1,  DI                  ,"jgtu  ",
  2,1,  DI                  ,"jle   ",
  2,1,  DI                  ,"jc    ",/* 0DB */
  2,1,  DI                  ,"jvt   ",
  2,1,  DI                  ,"jv    ",
  2,1,  DI                  ,"jlt   ",
  2,1,  DI                  ,"je    ",/* 0DF */
  3,1,  DI                  ,"djnz  ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",/* 0E3 */
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  3,1,  DI                  ,"jump  ",/* 0E7 */
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",/* 0EB */
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  3,1,  DI                  ,"lcall ",/* 0EF */
  1,0,      ID              ,"ret   ",
  1,0,  DI                  ,"reti  ",
  1,0,  DI                  ,"pushp ",
  1,0,  DI                  ,"popp  ",/* 0F3 */
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",/* 0F7 */
  1,0,  DI                  ,"clc   ",
  1,0,  DI                  ,"stc   ",
  1,0,  DI                  ,"di    ",
  1,0,  DI                  ,"ei    ",/* 0FB */
  1,0,  DI                  ,"clrvt ",
  1,0,UK                    ,"      ",
  1,0,UK                    ,"      ",
  1,0,  DI                  ,"nop   ",/* 0FF */

    0,0,0,"\0\0\0\0"};    /* end  */




/***************************************************************
 *  interpret a range of PC's
 *
 *  returns # of bytes interpreted for output line
 ***************************************************************/

long    intr_rang(  int         typ,        /* type of interpretation   */
                    char        *ibuf,      /* buffer to interpret      */
                    unsigned    pc,         /* current PC               */
                    long        inpcnt,     /* # of bytes to interpret  */
                    char        labels[],   /* labels array             */
                    COMTBLE     cmts[],     /* comments                 */
                    int         no_cmts,    /* no of comments in array  */
                    int         show_hex,   /* show hex data            */
                    FILE        *fpot)      /* output FILE              */

{   int         sts, i, cnt, cmtx;
    long        inpx;
    char        *label, *p, *leading, outbuf[200];
    COMTBLE     *cmt;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};
    static char blanks[]
          = "                                               ";
    leading = show_hex ? "" : "";
/* find position in comment array                   */
    for(cmtx = 0; cmtx < no_cmts; ++cmtx)
        if(pc <= cmts[cmtx].pc)
            break;

/* process range requested                          */
    for(inpx = 0; inpx < inpcnt; )
    {
/* display any line comments for the current line   */
        for( ; cmtx < no_cmts; ++cmtx)
        {   cmt = &cmts[cmtx];
            if(pc < cmt->pc)        /* no more for this line              */
                break;
            if(pc > cmt->pc)        /* was skipped, doesn't align       */
                fprintf(fpot, "%s**** Comment not aligned, PC=%x, %s\n",
                                                leading, cmt->pc, cmt->comnt);
            else                    /* this comment for this line         */
            {   if(cmt->typ == 1)           /* all line comments done   */
                    break;
                else if(cmt->typ == 2)      /* this is a line comment   */
                    fprintf(fpot, "%s%s\n", leading, cmt->comnt);
            }
        }
    
/* display any labels present for the current line  */
        while(0 <= (sts = next_name(pc, &label)))
        {   strcpy(outbuf, leading);
            if(show_hex)
            {  if((sts = format_2bytes(pc, &outbuf[0])) < 0)
                    return(sts);
            }
            strcat(outbuf, label);
            fprintf(fpot, "%s\n", outbuf);
        }

/* determine how far to next label or comment     */

        
/* display a line of output                       */
        switch(typ)
        {   case    DIS_IGNORE:
                cnt = intr_ign_line(  &ibuf[inpx], pc, (unsigned) (inpcnt - inpx), outbuf);
                break;
            case    DIS_HEX:
                cnt = intr_hex_line(  &ibuf[inpx], pc, (unsigned) (inpcnt - inpx), outbuf);
                break;
            case    DIS_BYTES:
                cnt = intr_byte_line( &ibuf[inpx], pc, (unsigned) (inpcnt - inpx), outbuf);
                break;
            case    DIS_WORDS:
                cnt = intr_word_line( &ibuf[inpx], pc, (unsigned) (inpcnt - inpx), outbuf);
                break;
            case    DIS_ASCII:
                cnt = intr_ascii_line(&ibuf[inpx], pc, (unsigned) (inpcnt - inpx), outbuf);
                break;
            case    DIS_CODE:
                cnt = intr_code_line( &ibuf[inpx], pc, (unsigned) (inpcnt - inpx), outbuf);
                break;
            default:
                fprintf(stderr, "****Error: Bad Command Table entry\n");
                exit(1);
                break;
        }
        if(cnt < 0)
        {   fprintf(stderr, "****Error in interpret: %d\n", cnt);
            exit(1);
        }
    /* add a generated label if needed      */
        if(check_bit(labels, pc))
        {   p = &outbuf[20];
            *p++ = 'L';
            format_2bytes(pc, p);
        }

    /* remove trailing blanks from line     */
        for(i = strlen(outbuf); 0 < i; --i)
            if(outbuf[i - 1] != ' ')
                break;
        outbuf[i] = '\0';

    /* add end of line comments             */
        if(pc == cmt->pc)           /* comment present for this line        */
        {   if(cmt->typ == 1)           /* its an end of line comment       */
            {   i = strlen(outbuf);
                if(i < 45)                  /* fill out to 45               */
                    strcat(outbuf, blanks +i);
                else                        /* just add 4, if > 45 now      */
                    strcat(outbuf, "    ");
                strcat(outbuf, cmt->comnt);
                ++cmtx;             /* over this one                        */
            }
            i= cmtx;                /* skip any other comments for this PC  */
            while(cmtx < no_cmts)
            {   cmt = &cmts[cmtx];
                if(pc == cmt->pc)
                    ++cmtx;
                else
                    break;
            }
            if(i != cmtx)
                fprintf(stderr, "**** ERROR, %d comments skipped at PC=%4.4x\n", cmtx - i, pc);
        }
    /* print 1 line from 'intr' routine     */
        if(!show_hex)                       /* dont' show 1st 20 columns    */
        {   if(strlen(outbuf) < 20)             /* rec is short, use null   */
                p = "";
            else                                /* normal, skip 1st 20      */
                p = &outbuf[20];
        }
        else                                /* show hex output              */
            p = outbuf;
        fprintf(fpot, "%s\n", p);           /* print the line               */
 
     /* incr counts and loop                 */        
        pc   += cnt;
        inpx += cnt;
    }
    return(inpx);
}

               



/***************************************************************
 *  interpret a line of output as 'ignore'
 *
 *  returns # of bytes interpreted for output line
 ***************************************************************/
int intr_ign_line(  char        *buf,       /* buffer to interpret      */
                    unsigned    pc,         /* current PC               */
                    unsigned    inpcnt,     /* max # bytes              */
                    char        *outp)      /* output buffer            */
               
{   int         sts;

    memset(outp, ' ', 79);
    
  /* put PC in 1st 4 chars      */
    if((sts = format_2bytes(pc, &outp[0])) < 0)
        return(sts);

  /* put DS op-code in output   */
    memcpy(&outp[26], "ds", 2);
    outp[32] = '$';
    if((sts = format_2bytes(inpcnt, &outp[33])) < 0)
        return(sts);
    strcpy(&outp[47], "; bytes ignored ");

  /* return                     */
    return(inpcnt);
}               

               



/***************************************************************
 *  interpret a line of output as 'hex'
 *
 *  returns # of bytes interpreted for output line
 ***************************************************************/
int intr_hex_line(  char        *buf,       /* buffer to interpret      */
                    unsigned    pc,         /* current PC               */
                    unsigned    inpcnt,     /* max # bytes              */
                    char        *outp)      /* output buffer            */
               
{   int         i, sts, dig, cnt;
    char        *p;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

	if ( inpcnt > 8 )
		cnt = (8 -(pc & 0x0007));
	else
		cnt =  inpcnt;
    memset(outp, ' ', 79);
    outp[79] = '\0';
    
  /* put PC in 1st 4 chars      */
    if((sts = format_2bytes(pc, &outp[0])) < 0)
        return(sts);

  /* put hex object in output   */
    if((sts = format_nbytes(&buf[0], cnt < 8 ? cnt : 8, &outp[6])) < 0)
        return(sts);

  /* put op-code in output      */
    memcpy(&outp[26], "byte  ", 6);

  /* display hex characters     */
    p = &outp[32];
    for(i = 0; i < cnt; ++i)
    {   *p++ = '$';
        dig  = (buf[i] >> 4) & 0xf;     /* get first digit          */
        p[0] = hex[dig];                /* char. to display         */
        dig  = (buf[i]     ) & 0xf;     /* get 2nd digit            */
        p[1] = hex[dig];                /* char to display          */
        if(i < cnt - 1)                 /* comma, unless its last   */
            p[2] = ',';
        p += 3;
    }
    return(cnt);
}               
               



/***************************************************************
 *  interpret a line of output as 'byte'
 *
 *  returns # of bytes interpreted for output line
 ***************************************************************/
int intr_byte_line( char        *buf,       /* buffer to interpret      */
                    unsigned    pc,         /* current PC               */
                    unsigned    inpcnt,     /* max # bytes              */
                    char        *outp)      /* output buffer            */
               
{   int         i, sts, dig, cnt;
    char        *p;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

    cnt =  inpcnt < 1  ?  inpcnt  :  1;
    memset(outp, ' ', 79);
    outp[79] = '\0';
    
  /* put PC in 1st 4 chars      */
    if((sts = format_2bytes(pc, &outp[0])) < 0)
        return(sts);

  /* put hex object in output   */
    if((sts = format_nbytes(&buf[0], cnt < 4 ? cnt : 4, &outp[6])) < 0)
        return(sts);

  /* put op-code in             */
    memcpy(&outp[26], "byte  ", 6);

  /* display hex characters     */
    p = &outp[32];
    for(i = 0; i < cnt; ++i)
    {   *p++ = '$';
        dig  = (buf[i] >> 4) & 0xf;     /* get first digit          */
        p[0] = hex[dig];                /* char. to display         */
        dig  = (buf[i]     ) & 0xf;     /* get 2nd digit            */
        p[1] = hex[dig];                /* char to display          */
        if(i < cnt - 1)                 /* comma, unless its last   */
            p[2] = ',';
        p += 3;
    }

  /* return                     */
    return(cnt);
}               
               



/***************************************************************
 *  interpret a line of output as 'word'
 *
 *  returns # of bytes interpreted for output line
 ***************************************************************/
int intr_word_line( char        *buf,       /* buffer to interpret      */
                    unsigned    pc,         /* current PC               */
                    unsigned    inpcnt,     /* max # bytes              */
                    char        *outp)      /* output buffer            */
               
{   int         i, sts, cnt, len;
    unsigned    val;
    char        *p, *nam, revb[2];
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

  /* init output buffer         */
    memset(outp, ' ', 79);
    outp[79] = '\0';

  /* get # bytes to interpret   */
    cnt =  inpcnt < 2  ?  inpcnt  :  2;     /* max of 2 bytes   */
    cnt =  (cnt + 1) & ~1;                  /* must be even     */
    
  /* put PC in 1st 4 chars      */
    if((sts = format_2bytes(pc, &outp[0])) < 0)
        return(sts);

  /* put hex object in output   */
    if((sts = format_nbytes(&buf[0], cnt < 4 ? cnt : 4, &outp[6])) < 0)
        return(sts);

  /* put op-code in output      */
    memcpy(&outp[26], "word  ", 6);

  /* display words              */
    p = &outp[32];
    for(i = 0; i < cnt / 2; ++i)
    {   val = ((buf[i * 2 + 1] & 0xff) << 8) + (buf[i * 2] & 0xff);
        sts = find_name(val, &nam);  /* see if name is known                */
        if(0 <= sts)                        /* name is known, use it        */
        {   len = strlen(nam);
            memcpy(p, nam, len);
            p += len;
        }
        else                                /* name is not known, use hex   */
        {   revb[0] = (val >> 8) & 0xff;
            revb[1] = (val     ) & 0xff;
            *p++ = '$';
            if((sts = format_nbytes(revb, 2, p)) < 0)
                return(sts);
            len = 4;
        }
        if(i < cnt / 2 - 1)                 /* comma, unless its last       */
            p[len] = ',';
        p += len + 1;
    }

  /* return                     */
    return(cnt);
}               
               



/***************************************************************
 *  interpret a line of output as 'ascii'
 *
 *  returns # of bytes interpreted for output line
 ***************************************************************/
int intr_ascii_line(char        *buf,       /* buffer to interpret      */
                    unsigned    pc,         /* current PC               */
                    unsigned    inpcnt,     /* max # bytes              */
                    char        *outp)      /* output buffer            */
               
{   int         i, sts, ch, cnt;
    char        *p;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};
    
    cnt =  inpcnt < 32  ?  inpcnt  :  32;
    memset(outp, ' ', 79);
    outp[79] = '\0';
    
  /* put PC in 1st 4 chars      */
    if((sts = format_2bytes(pc, &outp[0])) < 0)
        return(sts);

  /* put 1st 4 hex bytes in output  */
    if((sts = format_nbytes(&buf[0], cnt < 4 ? cnt : 4, &outp[6])) < 0)
        return(sts);

  /* put op-code in output          */
    memcpy(&outp[26], "ascii ", 6);

  /* display ascii chars            */
    p = &outp[32];
    *p++ = '"';
    for(i = 0; i < cnt; ++i)
    {   ch = buf[i];
        if(ch < ' ' ||  '\x7f' <= ch)
            *p++ = '.';
        else
            *p++ = (char) ch;
    }
    *p++ = '"';

  /* return                     */
    return(cnt);
}               
               



/***************************************************************
 *  interpret a line of output as 'code'
 *
 *  returns # of bytes interpreted for output line
 ***************************************************************/
int intr_code_line( char        *cbuf,      /* buffer to interpret      */
                    unsigned    pc,         /* current PC               */
                    unsigned    inpcnt,     /* max # bytes              */
                    char        *outp)      /* output buffer            */
               
{   int         sts, cnt, opc, ocnt, oclen, shcnt, write;
    int         autoinc, longx, imm_word, prefix;
    unsigned    ofst, dest;
    char        *buf, *p, bitno;
    OPCTBLE     *ope;
    static char *ivopc  = ";**** Invalid Op Code ****";
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};
    
    buf = cbuf;
    memset(outp, ' ', 79);
    outp[79] = '\0';
    oclen = 6;

  /* get op-code entry and set prefix flag  */
    prefix = 0;
    if( (buf[0] & 0xff) == 0x00fe)          /* byte is a prefix code        */
    {   if(opctbl[buf[1] & 0xff].typ & PX)      /* opcode allows prefix     */
        {   prefix = 1;                                 /* set prefix flag  */
            ++buf;                                      /* to next byte     */
        }
    }
    opc = buf[0] & 0xff;
    ope = &opctbl[opc];

    write = ope->typ & WR;

  /* see if there's a long index byte       */
  /* used to add 1 to byte count if present */
    if((ope->typ & 3) == IX)    /* its an indexed instruction       */
        longx = buf[1] & 0x01;          /* get long indexed flag    */
    else                        /* its not an indexed instruction   */
        longx = 0;                      /* no extra bytes           */
    
  /* put PC in 1st 4 chars of output        */
    if((sts = format_2bytes(pc, &outp[0])) < 0)
        return(sts);

  /* put hex bytes in output                */
    cnt = ope->numbyts  +  longx + prefix;
    if((sts = format_nbytes(cbuf, cnt < 7 ? cnt : 7, &outp[6])) < 0)
        return(sts);

  /* put op-code in output                  */
    p = &outp[26];
    if(ope->typ & UK)           /* invalid op code                          */
    {   memcpy(p, "byte  $", 7);
        if((sts = format_nbytes(&buf[0], 1, &outp[33])) < 0)
            return(sts);
        memcpy(&outp[40], ivopc, strlen(ivopc));
    }
    else
        memcpy(p, ope->name, oclen);

    if(prefix)                  /* fix opcode if prefix present             */
    {   int     ocx;
        static  char *names[]   = { "mulu  ", "mul   ", "mulub ", "mulb  ",
                                    "divu  ", "div   ", "divub ", "divb  "  };
        for(ocx = 0; ocx < 5; ocx += 2)
        {   if(!memcmp(ope->name, names[ocx], oclen));
            {   memcpy(p, names[ocx + 1], oclen);
                break;
            }
        }
    }

  /* display operand/s                      */
    ocnt = ope->numops;
    p = &outp[32];

        /******* shifts *********************************/
    if(ope->typ & SH)
    {   shcnt = buf[1] & 0xff;      /* get shift count          */
      /** put register in output            */
        if((sts = format_register(buf[2], 1, p)) < 0)
            return(sts);
        p += sts;
        *p++ = ',';
      /** put shift count in output         */
        if(shcnt < 16)              /* its an immediate value   */
        {   if((sts = format_imm_op(&buf[1], 0, p)) < 0)
                return(sts);
            p    += sts;
        }
        else                        /* its in a register        */
        {   if((sts = format_register(buf[1], 0, p)) < 0)
                return(sts);
            p += sts;
        }
    }

        /******* LJMP and LCALL *************************/
    else if( (opc == 0xe7) || (opc == 0xef) )
    { /** put destination address in output */
        ofst = ((buf[2] << 8) & 0xff00) + (buf[1] & 0xff);
        if(ofst & 0x8000)       /* handle negative  */
            ofst |= ~0x7fff;
        dest = pc + 3 + ofst;   /* destination      */
        if((sts = format_address(dest, p)) < 0)
            return(sts);
        p += sts;
    }

        /******* SJMP and SCALL *************************/
    else if( (opc & 0xf0) == 0x20)
    { /** put destination address in output */
        ofst  = ((opc & 0x7) << 8) + (buf[1] & 0xff);
        if(ofst & 0x0400)       /* handle negative  */
            ofst |= ~0x03ff;
        dest = pc + 2 + ofst;   /* destination      */
        if((sts = format_address(dest, p)) < 0)
            return(sts);
        p += sts;
    }

        /******* conditional jumps **********************/
    else if( 0xd0 <= opc  &&  opc <= 0xdf)
    { /** put destination address in output */
        ofst  = buf[1] & 0xff;
        if(ofst & 0x0080)       /* make negative    */
            ofst |= ~0x00ff;
        dest = pc + 2 + ofst;   /* destination      */
        if((sts = format_address(dest, p)) < 0)
            return(sts);
        p += sts;
    }

        /******* DJNZ ***********************************/
    else if(opc == 0xe0)
    { /** put BREG in output                */
        if((sts = format_register(buf[1], 1, p)) < 0)
            return(sts);
        p += sts;
        *p++ = ',';

      /** put destination address in output */
        ofst  = buf[2];
        if(ofst & 0x80)         /* make negative    */
            ofst |= ~0xff;
        dest = pc + 3 + ofst;   /* destination      */
        if((sts = format_address(dest, p)) < 0)
            return(sts);
        p += sts;
    }

        /******* JBC and JBS    *************************/
    else if((opc & 0xf0) == 0x30)
    { /** put BREG in output                */
        if((sts = format_register(buf[1], 0, p)) < 0)
            return(sts);
        p += sts;
        *p++ = ',';

      /** put bit# in output                */
        bitno = opc & 0x7;
        if((sts = format_bitno(bitno, p)) < 0)
            return(sts);
        p += sts;
        *p++ = ',';

      /** put destination address in output */
        ofst  = buf[2];
        if(ofst & 0x80)         /* make negative    */
            ofst |= ~0xff;
        dest = pc + 3 + ofst;   /* destination      */
        if((sts = format_address(dest, p)) < 0)
            return(sts);
        p += sts;
    }

    else switch(ope->typ & 0x7)
    {
    /******************* direct ******************************/
        case    DI:
            switch(ocnt)
            {   case 1:
                    if((sts = format_register(buf[1], write, p)) < 0)
                        return(sts);
                    p    += sts;
                    break;
                case 2:
                            /**** destination register  ****/
                    if((sts = format_register(buf[2], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source register       ****/
                    if((sts = format_register(buf[1], 0, p)) < 0)
                        return(sts);
                    p += sts;
                    break;
                case 3:
                            /**** destination register  ****/
                    if((sts = format_register(buf[3], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source 1 register     ****/
                    if((sts = format_register(buf[2], 0, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source 2 register     ****/
                    if((sts = format_register(buf[1], 0, p)) < 0)
                        return(sts);
                    p += sts;
                    break;
            }
            break;

    /******************* IMMEDIATE ***************************/
        case    IM:
            imm_word = (ope->typ & IW) ?  1  :  0;
            switch(ocnt)
            {   case 1:
                    if((sts = format_imm_op(&buf[1], imm_word, p)) < 0)
                        return(sts);
                    p    += sts;
                    break;
                case 2:
                            /**** destination register  ****/
                    if((sts = format_register(buf[2 + imm_word], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source register       ****/
                    if((sts = format_imm_op(&buf[1], imm_word, p)) < 0)
                        return(sts);
                    p += sts;
                    break;
                case 3:
                            /**** destination register  ****/
                    if((sts = format_register(buf[3 + imm_word], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source 1 register     ****/
                    if((sts = format_register(buf[2 + imm_word], 0, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source 2 register     ****/
                    if((sts = format_imm_op(&buf[1], imm_word, p)) < 0)
                        return(sts);
                    p += sts;
                    break;
            }
            break;

    /******************* INDIRECT ****************************/
        case    ID:
            autoinc = buf[1] & 0x01;        /* auto increment flag  */
            switch(ocnt)
            {   case 1:
                    *p++  = '[';
                    if((sts = format_register(buf[1] & 0xfe, 0, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++ = ']';
                    if(autoinc)
                        *p++ = '+';
                    break;
                case 2:
                            /**** destination register  ****/
                    if((sts = format_register(buf[2], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source register       ****/
                    *p++  = '[';
                    if((sts = format_register(buf[1] & 0xfe, 0, p)) < 0)
                        return(sts);
                    p   += sts;
                    *p++ = ']';
                    if(autoinc)
                        *p++ = '+';
                    break;
                case 3:
                            /**** destination register  ****/
                    if((sts = format_register(buf[3], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source 1 register     ****/
                    if((sts = format_register(buf[2], 0, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source 2 register     ****/
                    *p++  = '[';
                    if((sts = format_register(buf[1] & 0xfe, 0, p)) < 0)
                        return(sts);
                    p   += sts;
                    *p++ = ']';
                    if(autoinc)
                        *p++ = '+';
                    break;
            }
            break;

    /******************* INDEXED *****************************/
        case    IX:
            switch(ocnt)
            {   case 1:
                    if((sts = format_idx_op(&buf[1], longx, p)) < 0)
                        return(sts);
                    p    += sts;
                    break;
                case 2:
                            /**** destination register  ****/
                    if((sts = format_register(buf[3 + longx], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source register       ****/
                    if((sts = format_idx_op(&buf[1], longx, p)) < 0)
                        return(sts);
                    p  += sts;
                    break;
                case 3:
                            /**** destination register  ****/
                    if((sts = format_register(buf[4 + longx], 1, p)) < 0)
                        return(sts);
                    p    += sts;
                    *p++  = ',';
                            /**** source 1 register     ****/
                    if((sts = format_register(buf[3 + longx], 0, p)) < 0)
                        return(sts);
                    p    += sts;

                    *p++  = ',';
                            /**** source 2 register     ****/
                    if((sts = format_idx_op(&buf[1], longx, p)) < 0)
                        return(sts);
                    p   += sts;
                    break;
            }
            break;
    }

  /* return                     */
    return(cnt);
}



/***************************************************************
 *  format bit # operand to print
 *  output is not a NULL-terminated string
 *
 *  returns:   #bytes put into output if successful
 *      OR    negative error code
 ***************************************************************/
int format_bitno(int    bitno,      /* bit # to format          */
                 char   *outp)      /* output buffer            */
               
{   char        *p;

    if(7 < bitno)
        fprintf(stderr, "**** Internal error, bit #:%d\n", bitno);
    p = outp;
    *p++ = 'B';
    *p++ = bitno + '0';
    return(p - outp);
}



/***************************************************************
 *  format register operand to print
 *  output is not a NULL-terminated string
 *
 *  returns:   #bytes put into output if successful
 *      OR    negative error code
 ***************************************************************/
int format_register(int         reg,        /* register # to format         */
                    int         write,      /* register is being written    */
                    char        *outp)      /* output buffer                */
               
{   int         dig, len, sts;
    char        *p, *nam;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

    p = outp;
/* see if name is known             */
    sts = find_name(reg, &nam);
/* name is known, use it            */
    if(0 <= sts)                    
    {   len = strlen(nam);
        memcpy(p, nam, len);
        p += len;
    }
/* name is not known, use hex       */
    else
    {   *p++ = 'R';
        dig  = (reg >> 4) & 0xf;        /* get first digit              */
        *p++ = hex[dig];                    /* char. to display         */
        dig  = (reg     ) & 0xf;        /* get 2nd digit                */
        *p++ = hex[dig];                    /* char to display          */
    }
/* return                           */
    return(p - outp);
}



/***************************************************************
 *  format an immediate operand to print
 *  output is not a NULL-terminated string
 *
 *  returns:   #bytes put into output if successful
 *      OR    negative error code
 ***************************************************************/
int format_imm_op(char  *bytep,     /* value to format          */
                  int   word_flg,   /* 0 => byte, 1 => word     */
                  char  *outp)      /* output buffer            */
               
{   int         val, dig;
    char        *p;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

    p    = outp;
    *p++ = '#';
/* get value                        */
    val  = bytep[0] & 0xff;
    if(word_flg)
        val += (bytep[1] << 8) & 0xff00;
/* display value                    */
    *p++ = '$';
    if(word_flg)
    {   dig  = (val >> 12) & 0xf;   /* get first digit of byte      */
        *p++ = hex[dig];                /* char. to display         */
        dig  = (val >>  8) & 0xf;   /* get 2nd digit of byte        */
        *p++ = hex[dig];                /* char to display          */
    }
    dig  = (val >> 4) & 0xf;        /* get first digit of byte      */
    *p++ = hex[dig];                    /* char. to display         */
    dig  = (val     ) & 0xf;        /* get 2nd digit of byte        */
    *p++ = hex[dig];                    /* char to display          */
/* return                           */
    return(p - outp);
}



/***************************************************************
 *  format an indexed operand to print
 *  output is not a NULL-terminated string
 *
 *  returns:   #bytes put into output if successful
 *      OR    negative error code
 ***************************************************************/
int format_idx_op(char  *bytep,     /* ptr to [xreg | ofst ..]  */
                  int   long_flg,   /* 0 => short, 1 => long    */
                  char  *outp)      /* output buffer            */
               
{   int         val, dig, sts;
    char        *p;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

    p    = outp;
/* get value                        */
    val  = bytep[1] & 0xff;
    if(long_flg)
        val += (bytep[2] << 8) & 0xff00;
/* print displacement               */
    *p++ = '$';
    if(long_flg)
    {   dig  = (val >> 12) & 0xf;   /* get first digit of byte      */
        *p++ = hex[dig];                /* char. to display         */
        dig  = (val >>  8) & 0xf;   /* get 2nd digit of byte        */
        *p++ = hex[dig];                /* char to display          */
    }
    dig  = (val >> 4) & 0xf;        /* get first digit of byte      */
    *p++ = hex[dig];                    /* char. to display         */
    dig  = (val     ) & 0xf;        /* get 2nd digit of byte        */
    *p++ = hex[dig];                    /* char to display          */
/* display index register value     */
    *p++ = '[';
    if((sts = format_register(bytep[0] & 0xfe, 0, p)) < 0)
        return(sts);
    p  += sts;
    *p++ = ']';

/* return                           */
    return(p - outp);
}




/***************************************************************
 *  format address to print
 *  output is not a NULL-terminated string
 *
 *  returns:  #bytes put into output if successful
 *      OR    negative error code
 ***************************************************************/
int format_address( unsigned    addr,       /* address to format        */
                    char        *outp)      /* output buffer            */
               
{   int         len, sts;
    char        *p, *nam;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

/* see if name is known                     */
    p = outp;
    sts = find_name(addr, &nam);
/* name is known, use it                    */
    if(0 <= sts)
    {   len = strlen(nam);
        memcpy(p, nam, len);
        p += len;
    }
/* name is not known, use generated label   */
    else
    {   *p++ = 'L';
        len = format_2bytes(addr, p);
        p += len;
    }
/* return                                   */
    return(p - outp);
}



/***************************************************************
 *  format 2 bytes as 4 hex digits to print
 *  output is not a NULL-terminated string
 *
 *  returns:   # characters in output  if successful
 *      OR    negative error code
 ***************************************************************/
int format_2bytes(  unsigned    val,        /* value to format          */
                    char        *outp)      /* output buffer            */
               
{   int         dig;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};
    
    dig  = (val >> 12) & 0xf;       /* get first digit          */
    *outp++ = hex[dig];                 /* put in display       */
    dig  = (val >>  8) & 0xf;       /* get 2nd digit            */
    *outp++ = hex[dig];                 /* put in display       */
    dig  = (val >>  4) & 0xf;       /* get 3rd digit            */
    *outp++ = hex[dig];                 /* put in display       */
    dig  = (val      ) & 0xf;       /* get 4th digit            */
    *outp++ = hex[dig];                 /* put in display       */

    return(4);
}



/***************************************************************
 *  format n bytes as 2*n hex digits to print
 *  output is not a NULL-terminated string
 *
 *  returns:   #bytes in output if successful
 *      OR    negative error code
 ***************************************************************/
int format_nbytes(  char    *val,       /* ptr to value to format   */
                    int     cnt,        /* # of bytes to format     */
                    char    *outp)      /* output buffer            */
               
{   int         i, dig;
    static char hex[16] = { '0','1','2','3','4','5','6','7',
                            '8','9','A','B','C','D','E','F'};

    for(i = 0; i < cnt; ++i)
    {   dig  = (val[i] >> 4) & 0xf;     /* get first digit              */
        *outp++ = hex[dig];                /* char. to display          */
        dig  = (val[i]     ) & 0xf;     /* get 2nd digit                */
        *outp++ = hex[dig];                /* char to display           */
    }
    return(2 * cnt);
}



/***************************************************************
 *  'process' an instruction for labels
 *
 *  If instruction is a jump of any sort, the bit in the label array is 
 *  set for the destination of the jump. 
 *
 *  returns length of instruction  if successful
 *    OR   negative error code
 ***************************************************************/
int proc_instr_labels(char      *ibuf,      /* buffer containing instr  */
                      unsigned  pc,         /* current PC               */
                      char      labels[])   /* label array to set       */
               
{   int         ilen, opc, ocnt, longx, prefix, jctyp;
    unsigned    ofst, dest;
    char        *buf;
    OPCTBLE     *ope;

    prefix = 0;
    longx  = 0;
    dest   = 0;

  /* set flag for prefix byte if present    */
    buf = ibuf;
    if( (buf[0] & 0xff) == 0x00fe)          /* byte is a prefix code        */
    {   if(opctbl[buf[1] & 0xff].typ & PX)      /* opcode allows prefix     */
        {   prefix = 1;                                 /* set prefix flag  */
            ++buf;                                      /* to next byte     */
        }
    }

  /* get opcode and opcode table entry ptr  */
    opc = buf[0] & 0xff;
    ope = &opctbl[opc];

  /* see if there's a long index byte       */
  /* used to add 1 to byte count if present */
    if((ope->typ & 3) == IX)    /* its an indexed instruction       */
        longx = buf[1] & 0x01;          /* get long indexed flag    */
    else                        /* its not an indexed instruction   */
        longx = 0;                      /* no extra bytes           */

  /* compute length of instruction          */
    ilen = ope->numbyts  +  longx + prefix;

  /* process operand/s                      */
    ocnt = ope->numops;

        /******* SJMP and SCALL   ***********************/
    if( (opc & 0xf0) == 0x20)
    { /** get destination address           */
        ofst  = ((opc & 0x7) << 8) + (buf[1] & 0xff);
        if(ofst & 0x0400)       /* make negative    */
            ofst |= ~0x03ff;
        dest  = pc + 2 + ofst;  /* destination      */
        jctyp = (opc & 0x08) ? JC_CALL : JC_JUMP;
        set_bit(labels, dest);
        add_jcent(dest, pc, jctyp);
    }

        /******* conditional jumps **********************/
    else if( 0xd0 <= opc  &&  opc <= 0xdf)
    { /** get destination address           */
        ofst  = buf[1] & 0xff;
        if(ofst & 0x0080)       /* make negative    */
            ofst |= ~0x00ff;
        dest  = pc + 2 + ofst;  /* destination      */
        set_bit(labels, dest);
        add_jcent(dest, pc, JC_COND);
    }

        /******* DJNZ             ***********************/
    else if(opc == 0xe0)
    { /** get destination address           */
        ofst  = buf[2];
        if(ofst & 0x80)         /* make negative    */
            ofst |= ~0xff;
        dest  = pc + 3 + ofst;  /* destination      */
        set_bit(labels, dest);
        add_jcent(dest, pc, JC_COND);
    }

        /******* JBC and JBS      ***********************/
    else if((opc & 0xf0) == 0x30)
    { /** get destination address           */
        ofst  = buf[2];
        if(ofst & 0x80)         /* handle negative  */
            ofst |= ~0xff;
        dest  = pc + 3 + ofst;  /* destination      */
        set_bit(labels, dest);
        add_jcent(dest, pc, JC_COND);
    }

        /******* LJMP & LCALL     ***********************/
    else if(opc == 0xe7  ||  opc == 0xef)
    {   ofst = ((buf[2] << 8) & 0xff00) + (buf[1] & 0xff);
        if(ofst & 0x8000)       /* handle negative  */
            ofst |= ~0x7fff;
        dest = pc + 3 + ofst;   /* destination      */
        jctyp = (opc == 0xef) ? JC_CALL : JC_JUMP;
        set_bit(labels, dest);
        add_jcent(dest, pc, jctyp);
    }

        /******* BR[]             ***********************/
        /* shouldn't be in 806x's                       */
    else if(opc == 0xe3)
    {
        set_bit(labels, dest);
        add_jcent(dest, pc, JC_COND);
    }

  /* return                     */
    return(ilen);
}
